
// Define constants
const double precision = 1e-7; // a parameter for numerical integration, reduce to speed up calculation
const double pi = 3.14159265358979323846;

// Fit constants for protons in H2O
const double mp      = 1.007276466621; //in u
const double alpha_P = 0.02543;
const double p_P     = 1.742;
const double beta    = 0.0012;
const double gamma_H2O = 0.6;
const double epsilon = 0.;
const double rho     = 1.0e-3;
const double S       = 10000;

// Fit constants for helium/carbon in H2O
const double alpha_He = 0.00228;
const double p_He = 1.742;
const double alpha_C = 0.000222;
const double p_C = 1.625;

double alpha = alpha_P;
double p = p_P;

const double RSPHIT = 1.027; //1.02886;
vector<double> sheetThicknesses = {2.66,2.64,2.60,2.61,2.62,2.58,1.93,2.75,2.74,1.97,2.63,2.55,1.98,2.03,1.94,2.57,2.03,2.00,2.05,2.56,2.02,1.95,2.63,2.64,2.57,
    2.01,2.03,2.00,1.95,2.13,2.83,2.83,2.80,2.84,2.87,2.90,2.82,2.81,2.88,2.86,2.77,2.82,2.83,2.77,2.84,2.83,2.83,3.09,2.85,2.78,2.95,2.96,2.93,2.93,2.92};

double GetLinApproxPosition(TH1D* h, double value) {
    int binL = h->FindLastBinAbove(value);
    double yL = h->GetBinContent(binL);
    double xL = h->GetBinCenter(binL);
    int binR = binL+1;
    double yR = h->GetBinContent(binR);
    double xR = h->GetBinCenter(binR);
    double m = (yR-yL)/(xR-xL);
    double b = yL-m*xL;
    double xV = (value-b)/m;
    return xV;
}

//Estimate range of proton beam
double EstimateR0(TH1D *h){
  double maxValue = h->GetMaximum();
  double r0 = GetLinApproxPosition(h,0.8*maxValue);
  return r0;
}

//get dL/dz as TF1 without normalisation
double dLdz(double *zpp, double *par) {
    double R0    = par[0];
    double birk  = par[1];
    double dEdx_inv = p*pow(alpha,1./p)*pow((R0-zpp[0]),(1.-1./p));
    double val = 1./(dEdx_inv + birk);
    return val;
}

//get dL/dz without normalisation
double dLdz(double zp, double R0, double birk) {
    double dEdx_inv = p*pow(alpha,1./p)*pow((R0-zp),(1.-1./p));
    double val = 1./(dEdx_inv + birk);
    return val;
}

// Return term which includes dL/dz
double dLTerm(double *zp, double *par) {
    double R0     = par[0];
    double sigma  = par[1];
    double birk   = par[2];
    double z      = par[3];
    double preFactor = (1.+beta*(R0-zp[0]))/(1+beta*R0);
    double foldingTerm = 1./(sqrt(2*pi)*sigma)*exp(-(z-zp[0])*(z-zp[0])/(2*sigma*sigma));
    double val = preFactor * dLdz(zp[0],R0,birk) * foldingTerm;
    return val;
}

// Return term which includes L(z)
struct LTerm {
    TF1* fdLdz = nullptr;

    LTerm(TF1 &f) : fdLdz(&f) {}

    double operator() (double *zp, double *par) {
        double R0   = par[0];
        double birk = par[1];

        fdLdz->SetParameters(R0,birk);
        double LofZ = 0;
        if(zp[0]<(1-precision)*R0) LofZ = fdLdz->Integral(zp[0],(1-precision)*R0,precision);

        double preFactor = gamma_H2O*beta/(1+beta*R0);
        double foldingTerm = 1.;//approximation without folding. It is absolutely identical to with folding since LTerm is pretty flat

        double val = preFactor * LofZ * foldingTerm;
        return val;
    }
};

//This returns the actual quenched Bragg fit curve
struct QuenchedBragg {
    
    std::unique_ptr<TF1> fdLTerm;
    std::unique_ptr<TF1> fLTerm;
    std::unique_ptr<TF1> fdLdz;
    
    QuenchedBragg() {
        fdLdz   = std::unique_ptr<TF1>(new TF1("fdLdz",dLdz,0.,1.,2));
        fdLTerm = std::unique_ptr<TF1>(new TF1("fdLTerm",dLTerm,0.,1.,4));
        fLTerm  = std::unique_ptr<TF1>(new TF1("fLTerm",LTerm(*fdLdz),0.,1.,2));
    }
      
    double operator() (double* z, double* par) {
        double R0      = par[0];
        double sigma   = par[1];
        double phi0    = par[2];
        double xOffset = par[3];
        double birks   = par[4];

        if (R0<=0.) return 0.;
        if (sigma<=0.) return 0.;
        if (phi0<=0.) return 0.;
        if (xOffset<0.) return 0.;
        if (birks<=0.) return 0.;
        if (R0 != R0) return 0.; //Check if R0 is nan
        if (z[0] != z[0]) return 0.; //Check if z is nan
        z[0] = z[0] + xOffset; //add xOffset
        if (z[0] < 0) return 0.;

        // Some numerical parameters. They are optimized in such a way that the fit returns accurate results without throwing segfaults or unreached precision errors
        double zDef = 5.*sigma; //This constant determines in which area the integrator of fInt1 and fInt2 gonna be defined and integrated, see definition of a and b
        double a = min(z[0]-zDef,(1.-precision-1e-3)*R0); // integration start. Subtract constant to make sure a is smaller than b
        double b = min(z[0]+zDef,(1.-precision)*R0); // integration end

	fdLTerm->SetParameters(R0,sigma,birks,z[0]);
	double integral1 = fdLTerm->Integral(a,b,precision);

	fLTerm->SetParameters(R0,birks);
	double integral2 = fLTerm->Eval(z[0]); // approximation without folding
	double preFactor = S*phi0/rho;           // Put all parts together and add scaling factor
	double integral = integral1 + integral2; // Put all parts together and add scaling factor
        // Combine prefactor and integral of all parts
	double val = preFactor*integral;
        return val;
    }
};

struct testf{
  double operator() (double*t, double* par) {

    //gaussian function for contribution from electrostatic interactions
    double R0     = par[0];
    double sigma  = par[1];
    double z      = par[2];
    //    double val = exp(-(z-t[0])*(z-t[0])/(2*sigma*sigma))*p*pow(alpha,(1./p))*pow((R0-t[0]),(1./p-1));

    //    double val = par[0]*par[1]*t[0]+par[2];

    double val =  1/sqrt(2*pi*sigma*sigma)*exp(-1*((t[0]-z)*(t[0]-z))/(2*sigma*sigma));

    
    return val;
  }
};
