/**
 * @author: Saad Shaikh
 * @email: saad.shaikh.15@ucl.ac.uk
 * @purpose: Read, calibrate and fit photodiode data live
 * @usage: g++ -o liveFit liveFit.cpp `root-config --cflags --glibs` -O3 and then ./liveFit true
 */
#include <TSystem.h>
#include <TCanvas.h>
#include <TApplication.h>
#include <TF1.h>
#include <TH1.h>
#include <TAxis.h>
#include <TLegend.h>
#include <TLine.h>
#include <TStyle.h>
#include <TPaveText.h>
#include "Math/IntegratorOptions.h"
#include "Math/MinimizerOptions.h"
//#include "fitTools.h"
#include "fitTools_mod.h"

int main(int argc, char* argv[]) {
    
    ROOT::Math::IntegratorOneDimOptions::SetDefaultIntegrator("GaussLegendre");
    //    ROOT::Math::MinimizerOptions::SetDefaultMinimizer("Fumili");
    ROOT::Math::MinimizerOptions::SetDefaultMinimizer("Minuit");
    
    int num = 2;
    double FSR = 12.5;
    double yScale = 20;
    string background = "Run002_BackgroundNoLight_12-5pC_170us.txt";
    string ST_back = "Run003_270Shootthrough_12-5pC_170us.txt";
    string ST_front = "Run006_90Shootthrough_12-5pC_170us.txt";
    string showPlot = argv[1];
    bool bkgSubtract = false; //if significant electronic noise present in data
    vector<int> sheetOrder;
    vector<double> sheetEdges = {0.0}, data(photodiodes*num), dataErr(photodiodes*num);
    
    //Read stack configuration file
    ifstream fin("sheetConfig.txt");
    if (!fin.is_open()) {
        cout<<"ERROR: Cannot open sheet configuration file: sheetConfig.txt"<<endl;
        exit(-1);
    }
    string line, value;
    getline(fin,value); //read stack thickness
    double stackThickness = stod(value);
    getline(fin, line); //read sheet order
    stringstream ss(line);
    while(getline(ss, value, ',')) sheetOrder.push_back(stoi(value));
    getline(fin, value); //read stack WET
    double stackWET = stod(value);
    fin.close();
    
    //Calculate sheet edges in WET
    double sheetThicknessSum = 0, depth = 0;
    for (int i=0; i<sheetOrder.size(); i++) sheetThicknessSum += sheetThicknesses[sheetOrder[i]-1];
    if (stackWET == 0) stackWET = sheetThicknessSum*RSPHIT; //if no peakfinder value provided
    double RSP = stackWET/stackThickness;
    for (int i=0; i<sheetOrder.size(); i++) {
        depth += sheetThicknesses[sheetOrder[i]-1]*RSP; //in WET
        sheetEdges.push_back(depth);
    }
    
    //Read background data
    vector<vector<double>> backgroundData = readData(background, num, FSR);
    vector<vector<double>> backgroundStats = processData(backgroundData, num, FSR);
    backgroundData.clear();
    
    //Read calibration data (back)
    vector<vector<double>> backData = readData(ST_back, num, FSR);
    vector<vector<double>> backStats = processData(backData, num, FSR);
    backData.clear();
    
    //Read calibration data (front)
    vector<vector<double>> frontData = readData(ST_front, num, FSR);
    vector<vector<double>> frontStats = processData(frontData, num, FSR);
    frontData.clear();
    
    //Average and normalise shoot-throughs
    vector<double> caliMean(num*photodiodes), caliRMS(num*photodiodes);
    processST(caliMean, caliRMS, backStats, frontStats, num);

    //Set style options
    gStyle->SetOptStat(0000);
    gStyle->SetOptFit(01011);
    gStyle->SetStatW(0.1); //change W & H from 0.1 to 0 for PR plot
    gStyle->SetStatH(0.1);
    gStyle->SetPadTickX(kTRUE);
    gStyle->SetPadTickY(kTRUE);

    //Define main canvas
    if (showPlot == "true") TApplication* app = new TApplication("App",0,0);
    TCanvas* c1 = new TCanvas("c1","Quenched Bragg Fit",1200,750);
    TLegend* leg = new TLegend(0.8,0.59,0.98,0.69);
    TPad* pad1 = new TPad("pad1","pad1",0,0.3,1,1);
    pad1->SetBottomMargin(0); //joins upper and lower plot
    pad1->SetLeftMargin(0.11);
    TPad* pad2 = new TPad("pad2","pad2",0,0,1,0.3);
    pad2->SetTopMargin(0);
    pad2->SetBottomMargin(0.3);
    pad2->SetLeftMargin(0.11);
    
    //Residual plot
    TLine* l_res = new TLine(0.0,1,sheetEdges[photodiodes-1],1);
    if (showPlot == "true") {
        l_res->SetLineColor(kRed+1);
        l_res->SetLineWidth(lineWidth);
    }
    
    //FPS counter
    int count = 0;
    double sum = 0, fps = 0;
    string text = "0 FPS";
    TPaveText* pt_2 = new TPaveText(0.12,0.91,0.17,0.96,"ndc");
    if (showPlot == "true") {
        pt_2->AddText(text.c_str());
        pt_2->SetBorderSize(0);
        pt_2->SetTextAlign(12);
        pt_2->SetFillColor(kWhite);
        pt_2->SetTextFont(43);
        pt_2->SetTextSize(28);
    }

    //Energy label
    string E0text = "E_{0} = 0 MeV/u";
    TPaveText* pt = new TPaveText(0.8,0.55,0.85,0.58,"ndc");
    if (showPlot == "true") {
        pt->AddText(E0text.c_str());
        pt->SetBorderSize(0);
        pt->SetTextAlign(12);
        pt->SetFillColor(kWhite);
        pt->SetTextFont(43);
        pt->SetTextSize(24);
    }
    
    //Range label
    string R0text = "R_{0} = 0 mm";
    TPaveText* pt_4 = new TPaveText(0.8,0.50,0.85,0.53,"ndc");
    if (showPlot == "true") {
        pt_4->AddText(R0text.c_str());
        pt_4->SetBorderSize(0);
        pt_4->SetTextAlign(12);
        pt_4->SetFillColor(kWhite);
        pt_4->SetTextFont(43);
        pt_4->SetTextSize(24);
    }
    
    //Ratio label
    string Rtext = "0";
    TPaveText* pt_3 = new TPaveText(0.25,0.91,0.29,0.96,"ndc");
    if (showPlot == "true") {
        pt_3->AddText(text.c_str());
        pt_3->SetBorderSize(0);
        pt_3->SetTextAlign(12);
        pt_3->SetFillColor(kWhite);
        pt_3->SetTextFont(43);
        pt_3->SetTextSize(28);
    }
    
    double ratio_min = 1.8; //min data peak to average ratio to turn on fit
    QuenchedBragg qb; //TF1 Functors
    Bragg b;
    line.clear(); value.clear();
    auto start = chrono::high_resolution_clock::now();
    bool update = true;
    
    while (update) {
        fin.open("average.csv");
        if (fin.is_open()) {
            data.clear(); dataErr.clear();               //clear previous y-axis data
            int j=0;
            while(getline(fin, line, '\n')) {            //read row formatted data
                stringstream ss(line);
                while(getline(ss, value, ',')) {
                    if (j==0) data.push_back(stod(value));
                    else if (j==1) dataErr.push_back(stod(value));
                }
                j++;
            }
            fin.close();
            if (data.size() == photodiodes*num) {        //if full line is read
                count++;
                //Create histogram and place photodiode data
                TH1D* h_data = new TH1D("Photodiode Data","Quenched Bragg Fit",data.size(),sheetEdges.data());
                double avg = 0, peak = 0;
                for (int i=0; i<data.size(); i++) {
                    avg += data[i];
                    if (data[i] > peak) peak = data[i];
                    if (bkgSubtract) { //calibrate and subtract background
                        h_data->SetBinContent(i+1,data[data.size()-1-i]-backgroundStats[0][i]/caliMean[i]); //reverse order, subtract background and calibrate
                        h_data->SetBinError(i+1,(data[data.size()-1-i]-backgroundStats[0][i]/caliMean[i])*sqrt(pow(sqrt(pow(dataErr[data.size()-1-i],2)+pow(backgroundStats[1][i],2))/data[data.size()-1-i]-backgroundStats[0][i],2)+pow(caliRMS[i]/caliMean[i],2)));
                    }
                    else { //calibrate only
                        h_data->SetBinContent(i+1,data[data.size()-1-i]/caliMean[i]); //reverse order and calibrate
                        h_data->SetBinError(i+1,(data[data.size()-1-i]/caliMean[i])*sqrt(pow(dataErr[data.size()-1-i]/data[data.size()-1-i],2)+pow(caliRMS[i]/caliMean[i],2)));
                    }
                }
                avg = avg/data.size(); //Calculate ratio of peak value to average value, to determine if Bragg curve present
                double ratio = peak/avg;
                
                //Update parameters and fit
                double R0 = EstimateR0(h_data);
                double E0 = pow(R0/alpha,1/p)/mp;
                double xOffset = 0.0;
                double sigma_mono = 0.012*pow(R0,0.935);
                double sigma_E0 = 0.002*E0;                                         //in MeV. realistic initial beam energy spread?
                double sigma_0 = pow(sigma_E0*alpha*p,2)*pow(E0,2*p-2);             //in mm, estimated offset due to detector effects and initial beam energy spread
                double sigma = sqrt(pow(sigma_mono,2) + pow(sigma_0,2));
                double phi0 = 1e-7*h_data->GetBinContent(h_data->GetMaximumBin());  //rough guess
                double fit_start = 0;
                if(R0 < 30) fit_start = (R0-xOffset-7)/1.8;                         // protons hit sensor directly at low beam energies/ranges
                double fit_end = h_data->GetXaxis()->GetBinLowEdge(h_data->GetMaximumBin()+3); //light back to zero normally 3 sheets after peak
                double kb = 0.07;
                
                //Quenched Bragg Curve
                TF1* fphotons = new TF1("Fitted Light Output",&qb,fit_start,fit_end,5);
                fphotons->SetParameters(R0,sigma,phi0,xOffset,kb);
                fphotons->SetParNames("R_{0}","#sigma_{R}","#Phi_{0}","xOffset","kB");
                fphotons->FixParameter(3,xOffset);
                fphotons->SetNpx(100);
                if (ratio > ratio_min) h_data->Fit("Fitted Light Output","IRQ0"); //updates roughly 5x faster without I option
                else fphotons->SetParameters(0,0,0,0,0);                         //only fit if Bragg curve present
                fphotons->SetRange(0,sheetEdges[photodiodes*num]);               //expand range of fitted function to cover all photodiodes
                R0 = fphotons->GetParameter(0);                                  //update with fit results
                sigma = fphotons->GetParameter(1);
                E0 = pow(R0/alpha,1/p)/mp;
                
                //Bragg curve
                TF1* fBragg = new TF1("Reconstructed Bragg curve",&b,-xOffset,h_data->GetXaxis()->GetXmax(),4);
                fBragg->SetParameters(R0,sigma,1.,xOffset);
                fBragg->SetParNames("R_{0}","#sigma_{R}","#Phi_{0}","xOffset");
                double phi0_Bragg = fphotons->Eval(0)/fBragg->Eval(0);
                fBragg->SetParameter(2,phi0_Bragg);
                fBragg->SetNpx(100);
                
                if (showPlot == "true") {
                    h_data->SetLineColor(kBlack);
                    h_data->SetLineWidth(lineWidth);
                    TAxis* y = h_data->GetYaxis();
                    y->SetTitleOffset(titleOffset);
                    y->SetTitleSize(textSize);
                    y->SetTitleFont(font);
                    y->SetLabelFont(font);
                    y->SetLabelSize(textSize);
                    y->SetTitle("Charge (pC)");
                    TAxis* x = h_data->GetXaxis();
                    x->SetTitleOffset(titleOffset+2); //subtract 2 for PR plot
                    x->SetTitleSize(textSize);
                    x->SetTitleFont(font);
                    x->SetLabelFont(font);
                    x->SetLabelSize(textSize);
                    x->SetTitle("WET (mm)");
                    fphotons->SetLineWidth(lineWidth);
                    fphotons->SetLineColor(kBlue);
                    fBragg->SetLineWidth(lineWidth);
                    fBragg->SetLineColor(kGreen+2);
                    
                    // Residual plot
                    TH1D* h_res = new TH1D(*h_data);
                    h_res->Divide(fphotons);
                    h_res->SetTitle("");
                    h_res->GetYaxis()->SetTitle("PDL/QB");
                    h_res->GetYaxis()->SetNdivisions(204);
                    h_res->GetYaxis()->SetRangeUser(0.93,1.07);
                    h_res->GetXaxis()->SetTickLength(0.08);
                    l_res->SetX1(fit_start);
                    l_res->SetX2(fit_end);
                    
                    //Draw Histograms
                    y->SetRangeUser(-0.1,yScale); //1.7*h_data->GetBinContent(h_data->GetMaximumBin()));
                    x->SetLimits(0,sheetEdges[photodiodes-1]);
                    pad1->Draw();
                    c1->cd(); //returns to main canvas before defining pad2
                    pad2->Draw();
                    pad1->cd();
                    h_data->Draw("e");
                    fphotons->Draw("same");
                    fBragg->Draw("same");
                    pad2->cd();
                    h_res->Draw("e");
                    l_res->Draw("same");
                    pad1->cd();

                    //Update Legend
                    leg->Clear();
                    leg->AddEntry(h_data,"Measured PDL");
                    leg->AddEntry(fphotons,"Fitted Quenched Bragg Curve");
                    leg->AddEntry(fBragg,"Reconstructed Bragg Curve");
                    leg->Draw("same");

                    //Update Labels
                    E0text = "E_{0} = "+to_string(E0,1)+" MeV/u";
                    pt->Clear();
                    pt->AddText(E0text.c_str());
                    pt->Draw();
                    text = to_string(fps,2)+" FPS";
                    pt_2->Clear();
                    pt_2->AddText(text.c_str());
                    pt_2->Draw();
                    Rtext = to_string(ratio,2);
                    pt_3->Clear();
                    pt_3->AddText(Rtext.c_str());
                    pt_3->Draw();
                    R0text = "R_{0} = "+to_string(R0,1)+" mm";
                    pt_4->Clear();
                    pt_4->AddText(R0text.c_str());
                    pt_4->Draw();

                    //Update display
                    c1->Update();
                    gSystem->ProcessEvents();
                    delete h_res;
                }
                
                //Write averaged data and fit results to file
                ofstream out("output.csv");
                for (int i=0; i<num*photodiodes; i++) {
                    out << h_data->GetBinContent(i+1);
                    if (i<num*photodiodes-1) out << ", ";                    //add a comma for all but the last value
                    else out << "\n";
                }
                double increment = sheetEdges[photodiodes*num]/(10*num*photodiodes); //create 10 points per photodiode
                for (int i=0; i<10*num*photodiodes; i++) {
                    out << fphotons->Eval(i*increment);                      //write quenched Bragg points to file
                    if (i<10*num*photodiodes-1) out << ", ";                 //add a comma for all but the last value
                    else out << "\n";
                }
                for (int i=0; i<10*num*photodiodes; i++) {
                    out << fBragg->Eval(i*increment);                        //write Bragg points to file
                    if (i<10*num*photodiodes-1) out << ", ";                 //add a comma for all but the last value
                    else out << "\n";
                }
                out<<to_string(R0,1)<<"\n";
                out<<to_string(E0,1)<<"\n";
                out.close();
                
                //Clean up
                delete h_data;
                delete fphotons;
                
                //Calculate FPS
                auto end = chrono::high_resolution_clock::now();
                auto time = chrono::duration_cast<chrono::microseconds>(end - start);
                int elapsed = time.count();                //elapsed time in loop in us
                if (elapsed < 20000) {                     //If processing data faster than 50Hz
                    usleep(20000-elapsed);                 //Slow down to 50Hz
                    sum += 50;
                }
                else sum += (1/(double) elapsed)*1e6;
                fps = (double) sum/count;
                cout<<fps<<"\r"<<flush;
                start = chrono::high_resolution_clock::now();
            }
        }
    }
}
