/**
 * @author: Saad Shaikh
 * @email: saad.shaikh.15@ucl.ac.uk
 * @purpose: Store miscellaneous constants and functions
 */
#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <sstream>
#include <cstdlib>
#include <vector>
#include <map>
#include <cmath>
#include <chrono>
#include <thread>
#include <unistd.h>
using namespace std;

const int photodiodes = 16;

map<int,int> InputPhotodiodeMap {
    { 1, 1},
    { 2, 1},
    { 3, 2},
    { 4, 2},
    { 5, 3},
    { 6, 3},
    { 7, 4},
    { 8, 4},
    { 9, 5},
    {10, 5},
    {11, 6},
    {12, 6},
    {13, 7},
    {14, 7},
    {15, 8},
    {16, 8},
    {17, 9},
    {18, 9},
    {19, 10},
    {20, 10},
    {21, 11},
    {22, 11},
    {23, 12},
    {24, 12},
    {25, 13},
    {26, 13},
    {27, 14},
    {28, 14},
    {29, 15},
    {30, 15},
    {31, 16},
    {32, 16},
};

template <typename T>
string to_string(T const& value, int prec=2, int width=0) {
    stringstream sstr;
    sstr.precision(prec);
    if(width>0) sstr << std::setfill('0') << std::setw(width) << fixed << value;
    else sstr << fixed << value;
    return sstr.str();
}

string readLastLine(string filename, int lineNo) {
    string lastLine;
    ifstream fin;
    fin.open(filename);
    if(fin.is_open()) {
        int count = 0;
        fin.seekg(-2,ios_base::end);                      // Go to two spots before the EOF to ignore final "\n"
        bool keepLooping = true;
        while(keepLooping) {
            char ch;
            fin.get(ch);                                  // Get current byte's data
            if((int)fin.tellg() <= 1) {                   // If the data was at or before the 0th byte
                fin.seekg(0);                             // The first line is the last line
                keepLooping = false;                      // So stop there
            }
            else if(ch == '\n') {                         // If the data was a newline
                count++;                                  // Used to find the nth last line
                if (count == lineNo) keepLooping = false; // Stop at the current position.
                else fin.seekg(-2,ios_base::cur);         // Move back 2 chars
            }
            else {                                        // If the data was neither a newline nor at the 0 byte
                fin.seekg(-2,ios_base::cur);              // Move back 2 chars
            }
        }
        getline(fin,lastLine);                            // Read the current line
        fin.close();
    }
    else {
        cout << "ERROR: File cannot be opened: " << filename << endl;
        exit(-1);
    }
    return lastLine;
}

//Read photodiode data for noise analysis, either as individual inputs or as photodiodes
vector<vector<int>> readData(string filename, int num, bool channels) {
    vector<vector<int>> measurements;
    ifstream fin(filename);
    string line;
    int length;
    if (channels) length = 32*num;
    else length = photodiodes*num;
    if (fin.is_open()) {
        getline(fin,line);                                                //skip configuration data
        while(getline(fin,line)) {
            int module = num, iterator = 0;
            vector<int> measurement(length);
            for (int j=0; j<num; j++) {                                   //iterates over number of DDC232 boards
                module--;
                for (int i=0; i<32; i++) {                                //piece together 5-digit hex data for each channel
                    string input = "0x" + line.substr(iterator,5);
                    iterator += 5;
                    int value = stoi(input,nullptr,16);                   //convert hex to decimal
                    if (channels) measurement[32-i+(32*module)] = value;  //create record of measurement
                    else measurement[InputPhotodiodeMap[32-i]-1+(photodiodes*module)] += value;
                }
            }
            measurements.push_back(measurement);
        }
        fin.close();
    }
    else {
        cout << "ERROR: File cannot be opened: " << filename << endl;
        exit(-1);
    }
    return measurements;
}

//Read photodiode data for background subtraction/calibration/fitting, converting to pC
vector<vector<double>> readData(string filename, int num, double FSR) {
    vector<vector<double>> measurements;
    ifstream fin(filename);
    string line;
    int length = photodiodes*num;
    if (fin.is_open()) {
        getline(fin,line);                                                //skip configuration data
        while(getline(fin,line)) {
            int module = num, iterator = 0;
            vector<double> measurement(length);
            for (int j=0; j<num; j++) {                                   //iterates over number of DDC232 boards
                module--;
                for (int i=0; i<32; i++) {                                //piece together 5-digit hex data for each channel
                    string input = "0x" + line.substr(iterator,5);
                    iterator += 5;
                    double value = (stoi(input,nullptr,16)-4096)*(FSR/pow(2,20)); //convert hex to decimal, sub dark current & convert to charge
                    measurement[InputPhotodiodeMap[32-i]-1+(photodiodes*module)] += value;
                }
            }
            measurements.push_back(measurement);
        }
        fin.close();
    }
    else {
        cout << "ERROR: File cannot be opened: " << filename << endl;
        exit(-1);
    }
    return measurements;
}

//Calculate average and standard deviation in set of photodiode measurements
vector<vector<double>> processData(vector<vector<double>> data, int num, double FSR) {
    vector<double> mean(num*photodiodes), RMS(num*photodiodes);
    for (int j=0; j<num*photodiodes; j++) {
        int photodiode = num*photodiodes - j;                                //reverse order of photodiodes
        double avg = 0, stdDev = 0;
        for (int i=0; i<(int)data.size(); i++) avg += data[i][j];                 //calculate average photodiode value
        avg = avg/data.size();
        for (int i=0; i<(int)data.size(); i++) stdDev += pow(data[i][j] - avg,2); //calculate standard deviation of photodiode
        mean[photodiode-1] = avg;
        RMS[photodiode-1] = sqrt(stdDev/data.size())/sqrt(data.size());
    }
    vector<vector<double>> stats = {mean, RMS};
    return stats;
}

//Normalise and average front and back shoot-throughs
void processST(vector<double> &caliMean, vector<double> &caliRMS, vector<vector<double>> backStats, vector<vector<double>> frontStats, int num) {
    double norm, normRMS;
    double yShift = frontStats[0][(num*photodiodes/2)-1] - backStats[0][(num*photodiodes/2)-1]; //correct diff in absolute light
    for (int j=0; j<num*photodiodes; j++) {
        double mean = (frontStats[0][j] + yShift + backStats[0][j])/2;
        double RMS = sqrt(pow(frontStats[1][j],2)+pow(backStats[1][j],2))/2;
        if (j==0) {       //define first sheet as baseline
            norm = mean;
            normRMS = RMS;
        }
        caliMean[j] = mean/norm;
        caliRMS[j] = caliMean[j]*sqrt(pow(RMS/mean,2)+pow(normRMS/norm,2));
    }
}
