--
-- Buffer Controller Chip
-- Controller
--
-- Matt Warren
-- UCL
--
-- Provides all the BCC functions (that aren't command decoder)
--
--
-- Log:
-- 2009-Jan-07 - This file is born
-- 2009-Jan-12 - Updated for 4 channel readout
-- 2009-Jan-13 - Version 0.01 as sent to Dave Nelson
-- 2009-Jan-20 - Complete rework - Version 0.1 sent to Dave Nelson
-- 2009-Jan-21 - Tidied, removed rst
--             - Version 0.2
-- 2009-Jan-26 - Improved sensitivty lists for synth (why?)
--             - Version 0.3
-- 2009-Jan-27 - Version 0.4 (no changes since v0.3)
-- 2009-Feb-12 - rst now async
--             - gated asic bound signals with clk_locked
-- 2009-Feb-13 - Added not-sclking to data1,3 to ensure not transitions while
--               selected by mux.
--             - Version 0.5
-- 2009-Feb-16 - inverted reset, changed data mux modes to include not-sclk option
--             - removed clk_locked (it's now part of rstn)
--             - Version 0.6
-- 2009-Mar-9  - Added pd in data readout path to get better sim for mux
--             - Version 0.7



library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity bcc_control is
  port(
    abcn_dclk_o    : out std_logic;
    abcn_bco_o     : out std_logic;
    abcn_mode80_o  : out std_logic;
    abcn_com_o     : out std_logic;
    abcn_resetb_o  : out std_logic;
    abcn_l1_o      : out std_logic;
    local_com_i    : in  std_logic;
    local_l1_i     : in  std_logic;
    local_resetb_i : in  std_logic;
    data0_i        : in  std_logic;
    data1_i        : in  std_logic;
    data2_i        : in  std_logic;
    data3_i        : in  std_logic;
    local_data_o   : out std_logic;
    config_reg_i   : in  std_logic_vector (15 downto 0);
    comdest_i      : in  std_logic_vector (1 downto 0);
    rstn            : in  std_logic;
    clk            : in  std_logic;
    clk2x          : in  std_logic
    );

-- Declarations

end bcc_control;

--
architecture rtl of bcc_control is

  signal dclk : std_logic;
  signal sclk : std_logic;
  signal aclk : std_logic;

  signal data_sel  : std_logic;
  signal std_data  : std_logic;
  signal quad_data : std_logic;

  signal col0_data : std_logic;
  signal col1_data : std_logic;
  signal data_q    : std_logic_vector(3 downto 0);
  signal data_nq   : std_logic_vector(3 downto 0);

  signal com_com    : std_logic;
  signal com_l1     : std_logic;
  signal com_resetb : std_logic;


  signal constdata_mode : std_logic;
  signal col1_top_sel   : std_logic;
  signal col0_top_sel   : std_logic;
  signal aclk_en        : std_logic;
  signal abcn_dclk_en   : std_logic;
  signal abcn_bco_en    : std_logic;
  signal abcn_dclk_inv  : std_logic;
  signal abcn_bco_inv   : std_logic;
  signal aclk_inv       : std_logic;
  signal sclk_inv       : std_logic;
  signal mode80_sel     : std_logic;
  signal quadmode       : std_logic;
  signal linkmode       : std_logic_vector(1 downto 0);

  signal quadmode_selector : std_logic_vector(1 downto 0);


begin

  constdata_mode <= config_reg_i(13);
  col1_top_sel   <= config_reg_i(12);
  col0_top_sel   <= config_reg_i(11);
  abcn_dclk_en   <= config_reg_i(10);
  abcn_bco_en    <= config_reg_i( 9);
  aclk_en        <= config_reg_i( 8);
  abcn_dclk_inv  <= config_reg_i( 7);
  abcn_bco_inv   <= config_reg_i( 6);
  aclk_inv       <= config_reg_i( 5);
  sclk_inv       <= config_reg_i( 4);
  mode80_sel     <= config_reg_i( 3);
  quadmode       <= config_reg_i( 2);
  linkmode       <= config_reg_i(1 downto 0);


  abcn_bco_o  <= (clk xor abcn_bco_inv) and abcn_bco_en;
  abcn_dclk_o <= (dclk xor abcn_dclk_inv) and abcn_dclk_en;

  abcn_mode80_o <= mode80_sel;

  -- dclk selector
  dclk <= clk when (mode80_sel = '0') else clk2x;



  -- Signals TO asics section
  -------------------------------------------------------------------

  -- destination addressed signals  
  com_com    <= local_com_i      when comdest_i = "00" else '0';
  com_l1     <= local_com_i      when comdest_i = "01" else '0';
  com_resetb <= not(local_com_i) when comdest_i = "10" else '1';

  -- clock signals out to asics
  -- note: aclk_en is really just to keep aclk pd the same as bco

  aclk <= (clk xor aclk_inv) and aclk_en;

  prc_abcn_sigs_clk : process (aclk, rstn)
    -- com_com, local_l1_i, com_l1, rst, local_resetb_i, com_reset)
  begin
    if (rstn = '0') then
      abcn_com_o    <= '0';
      abcn_l1_o     <= '0';
      abcn_resetb_o <= '1';

    elsif rising_edge(aclk) then

        abcn_com_o    <= com_com;
        abcn_l1_o     <= local_l1_i or com_l1;
        abcn_resetb_o <= rstn and local_resetb_i and com_resetb;

    end if;
  end process;



  -- Data readout FROM asics section
  --------------------------------------------------------------

  sclk <= dclk xor sclk_inv;

  -- Clock data in
  prc_clock_data_in : process (sclk, rstn) --, data0_i, data1_i, data2_i, data3_i)
  begin
    if (rstn = '0') then
      data_q <= (others => '0');

    elsif rising_edge(sclk) then

      -- constdata_mode fixes link to known level 0,2 = lo, 1,3 = hi
      if (constdata_mode = '1') then
        data_q <= "1010";

      else
        data_q <= (data3_i & data2_i & data1_i & data0_i);

      end if;

    end if;
  end process;

  -- Clock data on the other edge to delay pulse across mux boundry
  prc_clock_data_nq : process (sclk, rstn)
  begin
    if (rstn = '0') then
      data_nq <= (others => '0');

    elsif falling_edge(sclk) then
      data_nq <= data_q;

    end if;
  end process;

  
  -- select redundant links
  col0_data <= data_q(0) after 600 ps  when (col0_top_sel = '0') else
					data_q(1) after 600 ps;
  
  col1_data <= data_nq(2) after 600 ps when (col1_top_sel = '0') else
					data_nq(3) after 600 ps;


  -- Standard Mode
  prc_data_linkmode : process (linkmode, sclk)
  begin
    case linkmode is
      when "00"   => data_sel <= sclk after 300 ps;
      when "01"   => data_sel <= not(sclk) after 600 ps;
      when "10"   => data_sel <= '0';
      when "11"   => data_sel <= '1';
      when others => null;
    end case;
  end process;

  std_data <= col0_data after 600 ps when (data_sel = '0') else
				  col1_data after 600 ps;

  -- Quad Mode
  quadmode_selector <= (clk & clk2x);

  prc_quadmode : process (quadmode_selector, data_q, data_nq)
  begin
    case quadmode_selector is
      when "00"   => quad_data <= data_q(0) after 900 ps;
      when "01"   => quad_data <= data_nq(2) after 900 ps;
      when "10"   => quad_data <= data_q(1) after 900 ps;
      when "11"   => quad_data <= data_nq(3) after 900 ps;
      when others => null;
    end case;
  end process;
 

  local_data_o <= quad_data when (quadmode = '1') else std_data;

end architecture rtl;
