--
-- Buffer Controller Chip
-- Command Decoder
--
-- Matt Warren
-- UCL
--
-- Decodes serial command stream and forwards it to bcc_control
-- if needed. Stores BCC config
--
--
-- Log:
-- 2009-Jan-07 - This file is born
-- 2009-Jan-12 - Updated with command option
-- 2009-Jan-13 - Version 0.01 as sent to Dave Nelson
-- 2009-Jan-20 - Complete rework version 0.1 sent to Dave Nelson
-- 2009-Jan-21 - Tidied, added ID readback via Type bits = 111.
--             - Version 0.2
-- 2009-Jan-26 - Fixed l1r decoding (and reworked decoder for no reason, bah!).
--             - Version 0.3
-- 2009-Jan-27 - Rework l1r decoding again - I preferred the original ;-)
-- 2009-Jan-27 - Found that a state-machine reseting itself makes latches - reworked
--             - Version 0.4
-- 2009-Jan-28 - reworked l1r decoder again, again (again?)
-- 2009-Feb-12 - made rst async (comes from por)
--             - moved l1r stuff to separate block
-- 2009-Feb-13 - Version 0.5
-- 2009-Feb-16 - inverted reset, moved l1r_reset to a higher level,
--               added por input for to resetting config register
--             - Version 0.6
-- 2009-Mar-09 - Version 0.7

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity bcc_decoder is
  port(
    id_i         : in  std_logic_vector (5 downto 0);
    com_i        : in  std_logic;
    data_o       : out std_logic;
    local_com_o  : out std_logic;
    local_data_i : in  std_logic;
    config_reg_o : out std_logic_vector (15 downto 0);
    comdest_o    : out std_logic_vector (1 downto 0);
    por          : in  std_logic;
    rstn         : in  std_logic;
    clk          : in  std_logic
    );

-- Declarations

end bcc_decoder;

--
architecture rtl of bcc_decoder is

  constant LEADER_LEN   : integer := 3;
  constant FASTCOM_LEN  : integer := 4;
  constant ADDRESS_LEN  : integer := 6;
  constant COMDEST_LEN  : integer := 3;
  constant CONFLEN_LEN  : integer := 16;
  constant READBACK_LEN : integer := 16;


  constant LEADER_COUNT  : integer := LEADER_LEN - 1;
  constant ADDRESS_COUNT : integer := ADDRESS_LEN + LEADER_COUNT;
  constant COMDEST_COUNT : integer := COMDEST_LEN + ADDRESS_COUNT;
  constant CONFLEN_COUNT : integer := CONFLEN_LEN + COMDEST_COUNT;

  -- fast command sub-machine
  constant FWDL1_COUNT      : integer := LEADER_LEN + LEADER_COUNT + 3;
  constant FWDFASTCOM_COUNT : integer := FASTCOM_LEN + LEADER_COUNT + 3;

  -- readback sub-machine
  constant READBACK_COUNT     : integer          := READBACK_LEN + COMDEST_COUNT + 1;
  constant BROADCASTID        : std_logic_vector := "111111";
  constant COMDEST_CONFIG     : std_logic_vector := "011";
  constant CONFIG_RESET_VALUE : std_logic_vector := "0000011100000001";


  type states is (s_WaitStart,
                  s_Leader,
                  s_Address,
                  s_ComDest,
                  s_Length,
                  s_Countdown,
                  s_Forward_L1,
                  s_Forward_FastCom,
                  s_Write_Config,
                  s_Readback_Config,
                  s_Readback_ID,
                  s_Readback_Transfer,

                  s_Reset
                  );

  signal state, current_state : states;

  signal sm_reset : std_logic;

  signal bit_count_clr    : std_logic;
  signal bit_count        : std_logic_vector(5 downto 0);
  signal address_ok_set   : std_logic;
  signal address_ok       : std_logic;
  signal sr_countdown_en  : std_logic;
  signal sr_counter       : std_logic_vector(15 downto 0);
  signal load_config_reg  : std_logic;
  signal config_reg       : std_logic_vector(15 downto 0);
  signal load_comdest_reg : std_logic;
  signal com_pipeline     : std_logic_vector(3 downto 0);
  signal com_out_en       : std_logic;
  signal delay_com_en     : std_logic;
  signal sr_load_config   : std_logic;
  signal sr_load_id       : std_logic;
  signal sr_sel           : std_logic_vector(2 downto 0);
  signal readback_out     : std_logic;
  signal readback_out_en  : std_logic;
  signal readback_end     : std_logic;

begin


  -- Decoder State Machine
  ---------------------------------------------------------------
  prc_machine_state : process (clk, rstn)
  begin

    if (rstn = '0') then
      current_state <= s_Reset;

    elsif rising_edge(clk) then
      current_state <= state;


    end if;
  end process;


  prc_sm_deserialise : process (current_state, sr_counter, bit_count, id_i, address_ok)
  begin

    -- defaults
    sm_reset         <= '0';
    bit_count_clr    <= '0';
    load_config_reg  <= '0';
    load_comdest_reg <= '0';
    address_ok_set   <= '0';
    sr_countdown_en  <= '0';
    com_out_en       <= '0';
    delay_com_en     <= '0';
    sr_load_config   <= '0';
    sr_load_id       <= '0';
    readback_out_en  <= '0';
    readback_end     <= '0';


    case current_state is

      when s_WaitStart =>
        bit_count_clr <= '1';
        state         <= s_WaitStart;   -- defaults

        if (sr_counter(0) = '1') then   -- start bit, clocked into shiftreg
          bit_count_clr <= '0';
          state         <= s_Leader;

        end if;


      when s_Leader =>
        state        <= s_Leader;
        delay_com_en <= '1';

        case (sr_counter((LEADER_LEN-1) downto 0)) is
          when "110"  => state <= s_Forward_L1;
          when "101"  => state <= s_Forward_FastCom;
          when "111"  => state <= s_Address;
          when others => state <= s_Leader;
        end case;


      when s_Address =>
        state <= s_Address;

        if (bit_count = ADDRESS_COUNT) then
          if (sr_counter((ADDRESS_LEN-1) downto 0) = id_i) then
            address_ok_set <= '1';
          end if;

          if (sr_counter((ADDRESS_LEN-1) downto 0) = BROADCASTID) then
            address_ok_set <= '1';
          end if;

          state <= s_ComDest;
        end if;


      when s_ComDest =>
        state <= s_ComDest;

        if (bit_count = COMDEST_COUNT) then
          load_comdest_reg       <= address_ok;
          case sr_counter(COMDEST_LEN-1 downto 0) is  -- msb = readback mode
            when "101"  => state <= s_Write_Config;
            when "100"  => state <= s_Readback_Config;
            when "111"  => state <= s_Readback_ID;
            when others => state <= s_Length;
          end case;
        end if;


      when s_Length =>
        state <= s_Length;

        if (bit_count = CONFLEN_COUNT) then
          sr_countdown_en <= '1';
          com_out_en      <= address_ok;
          state           <= s_Countdown;
        end if;


      when s_Countdown =>
        state <= s_Countdown;

        sr_countdown_en <= '1';
        com_out_en      <= '1';

        if (sr_counter = 0) then
          state <= s_Reset;
        end if;


      when s_Forward_L1 =>
        state <= s_Forward_L1;

        com_out_en   <= '1';
        delay_com_en <= '1';

        if (bit_count = FWDL1_COUNT) then
          state <= s_Reset;
        end if;


      when s_Forward_FastCom =>
        state <= s_Forward_FastCom;

        com_out_en   <= '1';
        delay_com_en <= '1';

        if (bit_count = FWDFASTCOM_COUNT) then
          state <= s_Reset;
        end if;


      when s_Write_Config =>
        state <= s_Write_Config;

        if (bit_count = CONFLEN_COUNT) then
          load_config_reg <= address_ok;
          state           <= s_Reset;
        end if;


      when s_Readback_Config =>
        sr_load_config <= address_ok;
        state          <= s_Readback_Transfer;


      when s_Readback_ID =>
        sr_load_id <= address_ok;
        state      <= s_Readback_Transfer;


      when s_Readback_Transfer =>
        state <= s_Readback_Transfer;

        readback_out_en <= address_ok;

        if (bit_count = READBACK_COUNT + 1) then  -- +1 to add trailbit
          readback_end <= '1';
          state        <= s_Reset;
        end if;


      when s_Reset =>
        sm_reset <= '1';
        state    <= s_WaitStart;


      when others =>
        state <= s_Reset;

    end case;
  end process;


  prc_machine_store : process (clk, rstn)
  begin
    if (rstn = '0') then
      address_ok <= '0';
      comdest_o  <= "00";

    elsif rising_edge(clk) then
      if (sm_reset = '1') then
        address_ok <= '0';
        comdest_o  <= "00";

      else
        if (address_ok_set = '1') then
          address_ok <= '1';
        end if;

        if (load_comdest_reg = '1') then
          -- cheaky way of using them msb of comdest to change output
          if (sr_counter(2) = '0') then
            comdest_o <= sr_counter(1 downto 0);
          else
            comdest_o <= "11";
          end if;

        end if;
      end if;

    end if;
  end process;


-- Bit Counter
-----------------------------------------------------------------
  prc_bit_counter : process (clk, rstn)
  begin
    if (rstn = '0') then
      bit_count <= (others => '0');

    elsif rising_edge(clk) then

      if ((sm_reset or bit_count_clr) = '1') then
        bit_count <= (others => '0');
      else
        bit_count <= bit_count + '1';
      end if;

    end if;
  end process;


-- Command shift register
--------------------------------------------------------------------

  sr_sel <= (sr_countdown_en & sr_load_config & sr_load_id);

  prc_sr_counter : process (clk, rstn)
  begin
    if (rstn = '0') then
      sr_counter <= (others => '0');

    elsif rising_edge(clk) then
      if (sm_reset = '1') then
        sr_counter <= (others => '0');

      else
        case sr_sel is
          when "100"  => sr_counter <= sr_counter - '1';
          when "010"  => sr_counter <= config_reg;
          when "001"  => sr_counter <= "0000000000" & id_i;
          when others => sr_counter <= sr_counter(14 downto 0) & com_i;

        end case;
      end if;
    end if;
  end process;




-- Config Register
------------------------------------------------------------------
  prc_config_reg : process (clk, rstn)
  begin
    if (por = '0') then  					 -- note: nor nrst, just por
      config_reg <= CONFIG_RESET_VALUE;

    else
      if rising_edge(clk) then
        if (load_config_reg = '1') then
          config_reg <= sr_counter;

        end if;
      end if;
    end if;
  end process;

  config_reg_o <= config_reg;




-- Data Out to BCC
----------------------------------------------------------------
  prc_readback_out : process (clk, rstn)
  begin
    if (rstn = '0') then
      readback_out <= '0';

    elsif rising_edge(clk) then
      readback_out <= (sr_counter(15) and readback_out_en)
                      or sr_load_config or sr_load_id or readback_end;  -- generate start/end bits on the cheap

    end if;
  end process;

  data_o <= local_data_i or readback_out;


-- Command passthrough pipeline
------------------------------------------------------------

-- bcc applies latency to command, allows routing
-- decisions to be made prior to forwarding

  prc_com_pipe : process (clk, rstn)
  begin
    if (rstn = '0') then
      com_pipeline <= (others => '0');
      local_com_o  <= '0';

    elsif rising_edge(clk) then

      com_pipeline <= com_pipeline(2 downto 0) & com_i;

      -- don't really need another FF here as com gets clocked out of the BCC,
      -- but this tidies things up a little

      if (delay_com_en = '1') then  	 -- only delay if doing bcast fast pass-through
        local_com_o <= com_pipeline(3) and com_out_en;

      else
        local_com_o <= com_i and com_out_en;

      end if;
    end if;
  end process;

end architecture rtl;
